<?php
// upload_account.php — smooth UI uploader for accounts
// Path: /bot/admin/upload_account.php

session_start();
if (!isset($_SESSION['admin'])) { header('Location: index.php'); exit; }

require __DIR__ . '/../db.php';
$config = require __DIR__ . '/../config.php';

$msg = '';
$err = '';

// fetch categories (all)
try {
    $cats = $pdo->query("SELECT id, title FROM categories ORDER BY sort_order, id")->fetchAll();
} catch (PDOException $e) {
    $cats = [];
    $err = "DB error: " . $e->getMessage();
}

// defaults for repopulate
$category_id     = (int)($_POST['category_id'] ?? 0);
$title           = trim($_POST['title'] ?? '');
$caption         = trim($_POST['caption'] ?? '');
$username_value  = trim($_POST['username_value'] ?? '');
$password_value  = trim($_POST['password_value'] ?? '');
$gmail           = trim($_POST['gmail'] ?? '');
$level           = trim($_POST['level'] ?? '');
$level_custom    = trim($_POST['level_custom'] ?? '');
$price_raw       = $_POST['price'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // compute real level (dropdown + optional custom)
    if ($level === 'other' && $level_custom !== '') {
        $final_level = $level_custom;
    } else {
        $final_level = $level !== '' ? $level : null;
    }

    $price = ($price_raw !== '') ? (float)$price_raw : null;

    // basic validation
    if ($title === '') {
        $err = "⚠️ Title is required.";
    } elseif ($category_id <= 0) {
        $err = "⚠️ Please choose a category.";
    } elseif (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        $err = "⚠️ Please choose an image to upload.";
    } else {
        $file = $_FILES['image'];
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $allowed = ['jpg','jpeg','png','gif','webp'];
        if (!in_array($ext, $allowed)) {
            $err = "⚠️ Invalid image type. Allowed: jpg, jpeg, png, gif, webp.";
        } else {
            $name = time() . "_" . bin2hex(random_bytes(6)) . "." . $ext;
            $dest_dir = __DIR__ . '/../uploads/';
            if (!is_dir($dest_dir)) @mkdir($dest_dir, 0755, true);
            $dest = $dest_dir . $name;

            if (!move_uploaded_file($file['tmp_name'], $dest)) {
                $err = "⚠️ Upload failed (server).";
            } else {
                // insert into DB
                try {
                    $stmt = $pdo->prepare(
                        "INSERT INTO accounts (category_id, title, caption, username_value, password_value, gmail, level, image_path, price)
                         VALUES (?,?,?,?,?,?,?,?,?)"
                    );
                    $stmt->execute([
                        $category_id,
                        $title,
                        $caption,
                        $username_value,
                        $password_value,
                        $gmail !== '' ? $gmail : null,
                        $final_level,
                        $name,
                        $price
                    ]);
                    $msg = "✅ Account uploaded successfully.";
                    // clear fields after success
                    $category_id = 0;
                    $title = $caption = $username_value = $password_value = $gmail = $level = $level_custom = $price_raw = '';
                } catch (PDOException $e) {
                    $err = "❌ DB error: " . $e->getMessage();
                    @unlink($dest);
                }
            }
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<title>Upload Account — Admin</title>
<meta name="viewport" content="width=device-width,initial-scale=1">

<style>
  :root{
    --bg:#050510;
    --card:#111119;
    --card-soft:#171724;
    --accent:#6c5ce7;
    --accent-soft:#a29bfe;
    --danger:#ff7675;
    --text-main:#ffffff;
    --text-muted:#a4a7b3;
    --border-soft:rgba(255,255,255,0.08);
  }
  *{box-sizing:border-box}
  body{
    margin:0;
    font-family:system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif;
    background:radial-gradient(circle at top,#1b1b33 0,#050510 55%);
    color:var(--text-main);
    -webkit-font-smoothing:antialiased;
  }
  .shell{
    max-width:980px;
    margin:0 auto;
    padding:18px 14px 40px;
  }
  .top-bar{
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap:12px;
    margin-bottom:16px;
  }
  .title-wrap h1{
    margin:0;
    font-size:22px;
    letter-spacing:.02em;
  }
  .subtitle{
    margin-top:4px;
    font-size:13px;
    color:var(--text-muted);
  }
  .nav{
    display:inline-flex;
    align-items:center;
    gap:8px;
    padding:4px;
    border-radius:999px;
    background:rgba(0,0,0,0.55);
    backdrop-filter:blur(12px);
  }
  .nav a{
    text-decoration:none;
    padding:8px 14px;
    border-radius:999px;
    font-size:13px;
    font-weight:600;
    display:inline-flex;
    align-items:center;
    gap:6px;
    background:linear-gradient(135deg,#ffffff,#f3f1ff);
    color:#181820;
    box-shadow:0 6px 18px rgba(0,0,0,0.32);
    border:1px solid rgba(255,255,255,0.85);
    transition:transform .12s ease, box-shadow .12s ease, background .12s ease;
  }
  .nav a:hover{
    transform:translateY(-1px);
    box-shadow:0 10px 26px rgba(0,0,0,0.38);
  }
  .nav a:active{
    transform:translateY(0);
    box-shadow:0 3px 10px rgba(0,0,0,0.25);
  }

  .card{
    background:rgba(15,15,28,0.96);
    border-radius:18px;
    padding:18px 16px 20px;
    border:1px solid var(--border-soft);
    box-shadow:0 18px 50px rgba(0,0,0,0.5);
  }
  .notice{padding:10px 12px;border-radius:10px;font-size:13px;margin-bottom:10px}
  .notice.ok{background:rgba(46,213,115,0.12);border:1px solid rgba(46,213,115,0.6);color:#b7ffce}
  .notice.err{background:rgba(255,71,87,0.12);border:1px solid rgba(255,71,87,0.7);color:#ffd4d8}

  .layout{
    display:flex;
    gap:18px;
    flex-wrap:wrap;
    align-items:flex-start;
  }
  .col-main{flex:1 1 320px}
  .col-side{width:280px;min-width:220px}

  .uploader{
    border-radius:14px;
    border:1px dashed rgba(255,255,255,0.2);
    padding:14px 12px;
    background:rgba(8,8,16,0.9);
    display:flex;
    flex-direction:column;
    gap:10px;
    align-items:center;
    cursor:pointer;
  }
  .uploader.dragover{
    border-style:solid;
    border-color:rgba(108,92,231,0.9);
    box-shadow:0 0 0 1px rgba(108,92,231,0.8);
  }
  .img-preview{
    width:100%;
    max-height:360px;
    border-radius:12px;
    overflow:hidden;
    background:linear-gradient(135deg,#1f2933,#111827);
    display:flex;
    align-items:center;
    justify-content:center;
  }
  .img-preview img{
    width:100%;
    height:100%;
    object-fit:cover;
    display:block;
  }
  .hint{color:var(--text-muted);font-size:13px;text-align:center;padding:16px 8px}

  .file-btn{
    border:none;
    border-radius:999px;
    padding:9px 16px;
    font-size:13px;
    font-weight:600;
    background:linear-gradient(120deg,var(--accent),var(--accent-soft));
    color:white;
    box-shadow:0 8px 22px rgba(108,92,231,0.55);
    cursor:pointer;
  }

  label{
    display:block;
    font-size:13px;
    color:#e5e7ff;
    margin-top:10px;
    margin-bottom:5px;
  }
  input[type=text],input[type=number],input[type=email],textarea,select{
    width:100%;
    padding:9px 10px;
    border-radius:10px;
    border:1px solid rgba(255,255,255,0.08);
    background:#050510;
    color:var(--text-main);
    font-size:14px;
  }
  textarea{min-height:110px;resize:vertical}
  input:focus,textarea:focus,select:focus{
    outline:none;
    border-color:var(--accent);
    box-shadow:0 0 0 1px rgba(108,92,231,0.65);
  }
  .muted{font-size:12px;color:var(--text-muted);margin-top:3px}
  .field-row{display:flex;gap:10px;flex-wrap:wrap}
  .field-row .col{flex:1 1 140px}

  .btn{
    border:none;
    border-radius:999px;
    padding:10px 16px;
    font-size:14px;
    font-weight:600;
    cursor:pointer;
    display:inline-flex;
    align-items:center;
    gap:6px;
    text-decoration:none;
  }
  .btn-primary{
    background:linear-gradient(120deg,var(--accent),var(--accent-soft));
    color:white;
    box-shadow:0 10px 26px rgba(108,92,231,0.55);
  }
  .btn-soft{
    background:rgba(255,255,255,0.06);
    color:var(--text-main);
    border:1px solid rgba(255,255,255,0.12);
  }
  .actions{display:flex;gap:10px;flex-wrap:wrap;margin-top:12px}

  .summary{
    background:var(--card-soft);
    border-radius:14px;
    padding:12px 12px 14px;
    border:1px solid var(--border-soft);
    font-size:13px;
  }
  .summary-title{font-weight:700;margin-bottom:6px}
  .summary-row{margin-top:4px;color:var(--text-muted)}
  .summary-row span{color:var(--text-main)}

  footer{
    margin-top:16px;
    text-align:center;
    font-size:13px;
    color:rgba(255,255,255,0.75);
  }

  @media(max-width:720px){
    .top-bar{flex-direction:column;align-items:flex-start}
    .nav{align-self:flex-end}
  }
</style>
</head>
<body>
<div class="shell">
  <div class="top-bar">
    <div class="title-wrap">
      <h1>Upload Account</h1>
      <div class="subtitle">Upload an image and fill the details. This will appear as a new account in your bot.</div>
    </div>
    <div class="nav">
      <a href="manage_categories.php">⚙️ Categories</a>
      <a href="accounts_list.php">📂 Accounts</a>
    </div>
  </div>

  <div class="card">
    <?php if ($msg): ?>
      <div class="notice ok"><?= htmlspecialchars($msg) ?></div>
    <?php endif; ?>
    <?php if ($err): ?>
      <div class="notice err"><?= htmlspecialchars($err) ?></div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data" id="uploadForm" novalidate>
      <div class="layout">
        <!-- LEFT: image + fields -->
        <div class="col-main">
          <!-- uploader -->
          <div id="uploader" class="uploader">
            <div id="imgPreview" class="img-preview">
              <div class="hint">No image selected yet</div>
            </div>

            <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center;justify-content:center;margin-top:8px">
              <button type="button" class="file-btn" id="fileBtn">Choose image</button>
              <input id="fileInput" name="image" type="file" accept="image/*" style="display:none" required>
              <span class="muted">Drop an image here or tap “Choose image”.</span>
            </div>
          </div>

          <!-- fields -->
          <label for="category_id">Category</label>
          <select name="category_id" id="category_id" required>
            <option value="">— choose category —</option>
            <?php foreach($cats as $c): ?>
              <option value="<?= (int)$c['id'] ?>" <?= $category_id == (int)$c['id'] ? 'selected' : '' ?>>
                <?= htmlspecialchars($c['title']) ?>
              </option>
            <?php endforeach; ?>
          </select>
          <div class="muted">Where this account will show inside the bot.</div>

          <label for="title">Title</label>
          <input type="text" name="title" id="title"
                 value="<?= htmlspecialchars($title) ?>"
                 placeholder="e.g. Gold Account — Level 20" required>

          <div class="field-row">
            <div class="col">
              <label for="level">Level</label>
              <select name="level" id="level">
                <option value="">— Select level —</option>
                <?php for($i=1;$i<=100;$i++): ?>
                  <option value="<?= $i ?>" <?= ($level == (string)$i) ? 'selected' : '' ?>><?= $i ?></option>
                <?php endfor; ?>
                <option value="other" <?= $level === 'other' ? 'selected' : '' ?>>Other / custom</option>
              </select>
            </div>
            <div class="col">
              <label for="level_custom">Custom level (optional)</label>
              <input type="text" name="level_custom" id="level_custom"
                     value="<?= htmlspecialchars($level_custom) ?>"
                     placeholder="e.g. Level 125 Asia">
            </div>
          </div>
          <div class="muted">Use dropdown for quick levels or type a custom value.</div>

          <label for="caption">Caption (under image)</label>
          <textarea name="caption" id="caption"
                    placeholder="Add details, skins, region, extra notes..."><?= htmlspecialchars($caption) ?></textarea>

          <div class="field-row">
            <div class="col">
              <label for="username_value">Username</label>
              <input type="text" name="username_value" id="username_value"
                     value="<?= htmlspecialchars($username_value) ?>"
                     placeholder="Account username">
            </div>
            <div class="col">
              <label for="password_value">Password</label>
              <input type="text" name="password_value" id="password_value"
                     value="<?= htmlspecialchars($password_value) ?>"
                     placeholder="Account password">
            </div>
          </div>

          <label for="gmail">Gmail (optional)</label>
          <input type="email" name="gmail" id="gmail"
                 value="<?= htmlspecialchars($gmail) ?>"
                 placeholder="owner@gmail.com">

          <label for="price">Price (optional)</label>
          <input type="text" name="price" id="price"
                 value="<?= htmlspecialchars($price_raw) ?>"
                 placeholder="e.g. 100.00">
          <div class="muted">Leave empty if you don't want to show a price.</div>

          <div class="actions">
            <button type="submit" class="btn btn-primary">➕ Upload Account</button>
            <a href="accounts_list.php" class="btn btn-soft">← Back to accounts</a>
          </div>
        </div>

        <!-- RIGHT: summary -->
        <div class="col-side">
          <div class="summary">
            <div class="summary-title">Preview summary</div>
            <div class="summary-row">Category: <span id="sumCategory">—</span></div>
            <div class="summary-row">Title: <span id="sumTitle">—</span></div>
            <div class="summary-row">Level: <span id="sumLevel">—</span></div>
            <div class="summary-row">Price: <span id="sumPrice">—</span></div>
            <div class="summary-row">Gmail: <span id="sumGmail">—</span></div>
            <div class="muted" style="margin-top:8px">
              This is how the key info will roughly look in Telegram after upload.
            </div>
          </div>
        </div>
      </div>
    </form>
  </div>

  <footer>⚡ Managed by your bot admin panel</footer>
</div>

<script>
  // elements
  const uploader   = document.getElementById('uploader');
  const fileInput  = document.getElementById('fileInput');
  const fileBtn    = document.getElementById('fileBtn');
  const imgPreview = document.getElementById('imgPreview');

  const catSel     = document.getElementById('category_id');
  const titleInput = document.getElementById('title');
  const priceInput = document.getElementById('price');
  const levelSel   = document.getElementById('level');
  const levelCustom= document.getElementById('level_custom');
  const gmailInput = document.getElementById('gmail');

  const sumCategory= document.getElementById('sumCategory');
  const sumTitle   = document.getElementById('sumTitle');
  const sumPrice   = document.getElementById('sumPrice');
  const sumLevel   = document.getElementById('sumLevel');
  const sumGmail   = document.getElementById('sumGmail');

  function setPreviewFromURL(url){
    imgPreview.innerHTML = '<img src="'+url+'" alt="preview">';
  }
  function setPreviewFromFile(file){
    const reader = new FileReader();
    reader.onload = e => setPreviewFromURL(e.target.result);
    reader.readAsDataURL(file);
  }

  // drag & drop
  ['dragenter','dragover'].forEach(ev=>{
    uploader.addEventListener(ev, e=>{
      e.preventDefault(); e.stopPropagation();
      uploader.classList.add('dragover');
    });
  });
  ['dragleave','drop'].forEach(ev=>{
    uploader.addEventListener(ev, e=>{
      e.preventDefault(); e.stopPropagation();
      uploader.classList.remove('dragover');
    });
  });
  uploader.addEventListener('drop', e=>{
    const f = e.dataTransfer.files && e.dataTransfer.files[0];
    if (!f) return;
    if (!f.type.startsWith('image/')) { alert('Please drop an image file.'); return; }
    handleFile(f);
  });

  fileBtn.addEventListener('click', ()=> fileInput.click());
  fileInput.addEventListener('change', e=>{
    const f = e.target.files && e.target.files[0];
    if (!f) return;
    handleFile(f);
  });

  async function handleFile(file){
    setPreviewFromFile(file);
    // small optional resize to max width 1200
    try{
      const imgBitmap = await createImageBitmap(file);
      const maxW = 1200;
      let w = imgBitmap.width, h = imgBitmap.height;
      if (w > maxW){
        const r = maxW / w;
        w = Math.round(w * r);
        h = Math.round(h * r);
      }
      if (w !== imgBitmap.width){
        const canvas = document.createElement('canvas');
        canvas.width = w; canvas.height = h;
        const ctx = canvas.getContext('2d');
        ctx.drawImage(imgBitmap,0,0,w,h);
        const type = file.type === 'image/png' ? 'image/png' : 'image/jpeg';
        const blob = await new Promise(res=>canvas.toBlob(res,type,0.9));
        if (blob && blob.size > 0){
          const newFile = new File([blob], file.name, {type: blob.type});
          const dt = new DataTransfer();
          dt.items.add(newFile);
          fileInput.files = dt.files;
          const url = URL.createObjectURL(newFile);
          setPreviewFromURL(url);
          setTimeout(()=>URL.revokeObjectURL(url),30000);
          return;
        }
      }
      const dt = new DataTransfer();
      dt.items.add(file);
      fileInput.files = dt.files;
    }catch(err){
      const dt = new DataTransfer();
      dt.items.add(file);
      fileInput.files = dt.files;
      console.warn('resize failed', err);
    }
  }

  // summary updates
  function updateCategorySummary(){
    if (!catSel || catSel.selectedIndex < 0) { sumCategory.textContent = '—'; return; }
    sumCategory.textContent = catSel.options[catSel.selectedIndex].text || '—';
  }
  function updateTitleSummary(){
    sumTitle.textContent = titleInput.value.trim() || '—';
  }
  function updatePriceSummary(){
    const v = priceInput.value.trim();
    sumPrice.textContent = v ? v : '—';
  }
  function updateLevelSummary(){
    if (levelSel.value === 'other') {
      sumLevel.textContent = levelCustom.value.trim() || 'Other';
    } else {
      sumLevel.textContent = levelSel.value || '—';
    }
  }
  function updateGmailSummary(){
    sumGmail.textContent = gmailInput.value.trim() || '—';
  }

  if (catSel)    { catSel.addEventListener('change', updateCategorySummary); updateCategorySummary(); }
  if (titleInput){ titleInput.addEventListener('input', updateTitleSummary); updateTitleSummary(); }
  if (priceInput){ priceInput.addEventListener('input', updatePriceSummary); updatePriceSummary(); }
  if (levelSel)  { levelSel.addEventListener('change', updateLevelSummary); updateLevelSummary(); }
  if (levelCustom){ levelCustom.addEventListener('input', updateLevelSummary); }
  if (gmailInput){ gmailInput.addEventListener('input', updateGmailSummary); updateGmailSummary(); }
</script>
</body>
</html>