<?php
// webhook.php — categories-enabled + nicer purchase/category formatting (includes level & gmail)
// Modified: BUY shows INR price, Direct Purchase owner link only on "More" row, "No more accounts" shows popup alert
// Added: Buy flow now sends UPI QR and inline "Verify Payment" button. Verify checks Paytm merchant-status API.
// Note: Requires phpqrcode library at __DIR__ . '/phpqrcode/qrlib.php' OR will fallback to sending UPI string as text.
require __DIR__ . '/db.php';
require __DIR__ . '/bot_functions.php';
$config = require __DIR__ . '/config.php';
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) exit;

// try include QR lib if present
$has_qrcode_lib = false;
if (file_exists(__DIR__ . '/phpqrcode/qrlib.php')) {
    require_once __DIR__ . '/phpqrcode/qrlib.php';
    if (class_exists('QRcode')) $has_qrcode_lib = true;
}

// ---------------- browse_sessions table (auto setup) ----------------
try {
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS browse_sessions (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            chat_id BIGINT NOT NULL,
            category_id INT NOT NULL,
            page INT NOT NULL DEFAULT 1,
            msg_ids TEXT NOT NULL,
            updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY uniq_user (user_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
} catch (PDOException $e) {
    error_log('DB ERROR browse_sessions create: ' . $e->getMessage());
}

// ---------------- orders table alter (for QR + error messages) ---------------
try {
    $pdo->exec("ALTER TABLE orders ADD COLUMN qr_message_id BIGINT NULL");
} catch (PDOException $e) {
    // probably already exists
}
try {
    $pdo->exec("ALTER TABLE orders ADD COLUMN last_error_message_id BIGINT NULL");
} catch (PDOException $e) {
    // probably already exists
}

// ---------------- helpers ----------------

// get owner username (if owner exists in users table)
function getOwnerUsername($pdo, $owner_tg_id) {
    try {
        $stmt = $pdo->prepare("SELECT username FROM users WHERE telegram_id = ? LIMIT 1");
        $stmt->execute([$owner_tg_id]);
        $r = $stmt->fetch();
        if ($r && !empty($r['username'])) return $r['username'];
    } catch (PDOException $e) { /* ignore */ }
    return null;
}

// format INR label: no decimals if whole number, else 2 decimals
function format_inr_label($price) {
    if ($price === null) return '';
    $p = (float)$price;
    if (floor($p) == $p) return '₹' . number_format($p, 0);
    return '₹' . number_format($p, 2);
}

// small wrapper to call Telegram API (used for answerCallbackQuery etc)
if (!function_exists('apiRequest')) {
    function apiRequest($method, $params = []) {
        global $config;
        $token = isset($config['bot_token']) ? $config['bot_token'] : null;
        if (!$token) return false;
        $url = "https://api.telegram.org/bot{$token}/{$method}";
        $ch = curl_init($url);
        $payload = json_encode($params);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $resp = curl_exec($ch);
        curl_close($ch);
        return $resp;
    }
}

// ---------------- browse session helpers (for auto-clean UX) ----------------

function getBrowseSession($pdo, $user_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM browse_sessions WHERE user_id = ? LIMIT 1");
        $stmt->execute([$user_id]);
        return $stmt->fetch();
    } catch (PDOException $e) {
        error_log("DB ERROR getBrowseSession: " . $e->getMessage());
        return null;
    }
}

function saveBrowseSession($pdo, $user_id, $chat_id, $category_id, $page, array $msg_ids) {
    $msg_ids = array_values(array_unique(array_map('intval', $msg_ids)));
    $json = json_encode($msg_ids);
    if ($json === false) $json = '[]';

    try {
        $stmt = $pdo->prepare("
            INSERT INTO browse_sessions (user_id, chat_id, category_id, page, msg_ids)
            VALUES (?,?,?,?,?)
            ON DUPLICATE KEY UPDATE
                chat_id = VALUES(chat_id),
                category_id = VALUES(category_id),
                page = VALUES(page),
                msg_ids = VALUES(msg_ids)
        ");
        $stmt->execute([$user_id, $chat_id, $category_id, $page, $json]);
    } catch (PDOException $e) {
        error_log("DB ERROR saveBrowseSession: " . $e->getMessage());
    }
}

/**
 * Delete all messages in current browse session for a user.
 * If $keep_message_id is provided, that message is NOT deleted and
 * the session is updated to only track that one message.
 * If $keep_message_id is null, session row is deleted.
 */
function cleanupBrowseSession($pdo, $user_id, $chat_id, $keep_message_id = null) {
    if (!$user_id) return;
    $session = getBrowseSession($pdo, $user_id);
    if (!$session) return;

    $raw = isset($session['msg_ids']) ? $session['msg_ids'] : '[]';

    if (is_string($raw)) {
        $ids = json_decode($raw, true);
        if (!is_array($ids)) $ids = [];
    } elseif (is_array($raw)) {
        $ids = $raw;
    } else {
        $ids = [];
    }

    foreach ($ids as $mid) {
        $mid = (int)$mid;
        if ($mid <= 0) continue;
        if ($keep_message_id !== null && (int)$keep_message_id === $mid) {
            continue;
        }
        apiRequest('deleteMessage', [
            'chat_id'    => $chat_id,
            'message_id' => $mid
        ]);
    }

    if ($keep_message_id === null) {
        try {
            $stmt = $pdo->prepare("DELETE FROM browse_sessions WHERE user_id = ?");
            $stmt->execute([$user_id]);
        } catch (PDOException $e) {
            error_log("DB ERROR cleanupBrowseSession delete: " . $e->getMessage());
        }
    } else {
        try {
            $json = json_encode([(int)$keep_message_id]);
            $stmt = $pdo->prepare("UPDATE browse_sessions SET msg_ids = ? WHERE user_id = ?");
            $stmt->execute([$json, $user_id]);
        } catch (PDOException $e) {
            error_log("DB ERROR cleanupBrowseSession update: " . $e->getMessage());
        }
    }
}

/**
 * Cancel & clean ALL pending payments UI for a user:
 * - delete QR message
 * - delete last "payment not received" message
 * - mark orders as cancelled
 */
function cleanupUserPendingPayments($pdo, $userRow) {
    if (!$userRow || !isset($userRow['id'])) return;
    $user_id = (int)$userRow['id'];
    $chat_id = isset($userRow['telegram_id']) ? $userRow['telegram_id'] : null;
    if (!$chat_id) return;

    try {
        $stmt = $pdo->prepare("SELECT id, qr_message_id, last_error_message_id FROM orders WHERE user_id = ? AND status = 'pending'");
        $stmt->execute([$user_id]);
        $rows = $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("DB ERROR cleanupUserPendingPayments fetch: " . $e->getMessage());
        return;
    }

    foreach ($rows as $r) {
        if (!empty($r['qr_message_id'])) {
            apiRequest('deleteMessage', [
                'chat_id'    => $chat_id,
                'message_id' => (int)$r['qr_message_id']
            ]);
        }
        if (!empty($r['last_error_message_id'])) {
            apiRequest('deleteMessage', [
                'chat_id'    => $chat_id,
                'message_id' => (int)$r['last_error_message_id']
            ]);
        }

        try {
            $upd = $pdo->prepare("UPDATE orders SET status='cancelled', qr_message_id=NULL, last_error_message_id=NULL, updated_at = NOW() WHERE id = ?");
            $upd->execute([$r['id']]);
        } catch (PDOException $e) {
            error_log("DB ERROR cleanupUserPendingPayments update: " . $e->getMessage());
        }
    }
}

// ---------------- existing helpers (unchanged-ish) ----------------

function getUserRow($pdo, $telegram_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegram_id]);
        return $stmt->fetch();
    } catch (PDOException $e) {
        error_log("DB ERROR getUserRow: " . $e->getMessage());
        return false;
    }
}
function createUserIfNotExists($pdo, $tg) {
    try {
        $u = getUserRow($pdo, $tg['id']);
        if ($u) return $u;
        $token = bin2hex(random_bytes(16));
        $username   = isset($tg['username']) ? $tg['username'] : null;
        $first_name = isset($tg['first_name']) ? $tg['first_name'] : null;
        $last_name  = isset($tg['last_name']) ? $tg['last_name'] : null;

        $stmt = $pdo->prepare("INSERT INTO users (telegram_id, username, first_name, last_name, token) VALUES (?,?,?,?,?)");
        $stmt->execute([$tg['id'], $username, $first_name, $last_name, $token]);
        return getUserRow($pdo, $tg['id']);
    } catch (PDOException $e) {
        error_log("DB ERROR createUserIfNotExists: " . $e->getMessage());
        return false;
    }
}
function isOwner($tg_id) {
    global $config;
    return (string)$tg_id === (string)$config['owner_chat_id'];
}

// helper: send user's purchased items (visible ones)
function sendUserPurchases($pdo, $userRow, $chat_id) {
    global $config;
    try {
        $stmt = $pdo->prepare("SELECT o.id as order_id, o.created_at as ordered_at, a.* 
                               FROM orders o
                               JOIN accounts a ON o.account_id = a.id
                               WHERE o.user_id = ? AND o.status = 'paid' AND o.visible = 1
                               ORDER BY o.created_at DESC");
        $stmt->execute([$userRow['id']]);
        $rows = $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("DB ERROR sendUserPurchases: ".$e->getMessage());
        sendMessage($chat_id, "⚠️ Internal error. Try again later.");
        return;
    }

    if (!$rows) {
        sendMessage($chat_id, "📭 You don't have any visible purchases right now.");
        return;
    }

    foreach ($rows as $r) {
        $img_url = $config['site_url'] . '/uploads/' . basename($r['image_path']);
        $title   = htmlspecialchars($r['title'] ?: 'Account');
        $level   = $r['level'] ? htmlspecialchars($r['level']) : null;
        $gmail   = $r['gmail'] ? htmlspecialchars($r['gmail']) : null;
        $cred_user = htmlspecialchars($r['username_value'] ?? '—');
        $cred_pass = htmlspecialchars($r['password_value'] ?? '—');
    
        $price_block = "";
        if (!is_null($r['price'])) {
            $p = number_format((float)$r['price'], 2);
            $price_block = "<blockquote><b>💰 Price:</b> <code>{$p}</code></blockquote>";
        }
    
        $lines = [];
        $lines[] = "<blockquote><b>🎉 {$title}</b></blockquote>";
        if ($level) {
            $lines[] = "<blockquote><b>📶 Level:</b> <code>{$level}</code></blockquote>";
        }
        $lines[] = "<blockquote><b>👤 Username:</b> <code>{$cred_user}</code></blockquote>";
        $lines[] = "<blockquote><b>🔑 Password:</b> <code>{$cred_pass}</code></blockquote>";
        if ($gmail) {
            $lines[] = "<blockquote><b>📧 Gmail:</b> <code>{$gmail}</code></blockquote>";
        }
        if ($price_block) {
            $lines[] = $price_block;
        }
        $lines[] = "<b>🙏 Thanks for purchasing — keep these credentials safe.</b>";
    
        $caption = implode("\n", array_filter($lines));
    
        $kb = [
            'inline_keyboard' => [
                [['text' => '🗑️ Remove', 'callback_data' => "remove_{$r['order_id']}"]]
            ]
        ];
    
        sendPhoto($chat_id, $img_url, $caption, $kb);
    }
}

// ---------------- incoming message ----------------

if (isset($input['message'])) {
    $msg     = $input['message'];
    $chat_id = $msg['chat']['id'];
    $from    = $msg['from'];
    $user    = createUserIfNotExists($pdo, $from);

    // /start — reset UX, clean previous stuff
    if (isset($msg['text']) && strpos($msg['text'], '/start') === 0) {

        if ($user && isset($user['id'])) {
            cleanupUserPendingPayments($pdo, $user);
            cleanupBrowseSession($pdo, $user['id'], $chat_id, null);
        }

        try {
            $cats = $pdo->query("SELECT id, title, slug FROM categories WHERE IFNULL(is_hidden,0) = 0 ORDER BY sort_order")->fetchAll();
        } catch (PDOException $e) {
            error_log("DB ERROR categories: " . $e->getMessage());
            $cats = [];
        }

        $inline = [];
        foreach ($cats as $c) {
            $inline[] = [['text' => "📁 " . $c['title'], 'callback_data' => "cat_{$c['id']}_page_1"]];
        }
        $reply_markup_inline = ['inline_keyboard' => $inline];

        $reply_markup2 = [
            'keyboard' => [
                [['text' => '📦 Show Purchased ids']]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];

        if (empty($inline)) {
            if (isOwner($chat_id)) {
                sendMessage($chat_id, "⚠️ No categories configured yet. Add categories via admin: /admin/manage_categories.php", $reply_markup2);
            } else {
                sendMessage($chat_id, "⏳ Shop is empty right now. Check back soon!", $reply_markup2);
            }
            exit;
        }

        sendMessage($chat_id, "✨ Welcome! Choose a category to browse available accounts:", $reply_markup2);
        sendMessage($chat_id, "📚 Categories:", $reply_markup_inline);
        exit;
    }

    if (isset($msg['text']) && ($msg['text'] === '📦 Show Purchased ids' || $msg['text'] === 'Show Purchased ids')) {
        $userRow = createUserIfNotExists($pdo, $msg['from']);
        if (!$userRow) {
            sendMessage($chat_id, "⚠️ Could not load your profile. Try /start.");
            exit;
        }
        sendUserPurchases($pdo, $userRow, $chat_id);
        exit;
    }

    exit;
}

// ---------------- callback_query handling ----------------

if (isset($input['callback_query'])) {
    $cb      = $input['callback_query'];
    $data    = $cb['data'] ?? '';
    $chat_id = $cb['message']['chat']['id'];
    $msg_id  = $cb['message']['message_id'];
    $from    = $cb['from'];
    $user    = createUserIfNotExists($pdo, $from);

    // category browse: cat_{id}_page_{n}
    if (preg_match('#^cat_(\d+)_page_(\d+)$#', $data, $m)) {
        $cat_id = (int)$m[1];
        $page   = (int)$m[2];
        $per    = 5;
        $offset = ($page - 1) * $per;

        if ($user && isset($user['id'])) {
            cleanupUserPendingPayments($pdo, $user);                  // delete QR + error msgs for pending
            cleanupBrowseSession($pdo, $user['id'], $chat_id, null);  // delete old cards + More
        }

        $perInt    = (int)$per;
        $offsetInt = (int)$offset;
        $sql = "SELECT * FROM accounts WHERE category_id = :cat AND is_sold=0 ORDER BY created_at DESC LIMIT {$perInt} OFFSET {$offsetInt}";

        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':cat' => $cat_id]);
            $rows = $stmt->fetchAll();
        } catch (PDOException $e) {
            error_log("DB ERROR fetch accounts: " . $e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Internal error.']);
            exit;
        }

        if (!$rows) {
            apiRequest('answerCallbackQuery', [
                'callback_query_id' => $cb['id'],
                'text' => '📭 No more accounts.',
                'show_alert' => true
            ]);
            exit;
        }

        $owner_username = getOwnerUsername($pdo, $config['owner_chat_id']);
        $owner_link = $owner_username ? "https://t.me/".urlencode($owner_username) : null;

        $sent_msg_ids = [];

        foreach ($rows as $acc) {
            $img_url = $config['site_url'] . '/uploads/' . basename($acc['image_path']);
            $title   = htmlspecialchars($acc['title'] ?: 'Account');
            $level   = $acc['level'] ? htmlspecialchars($acc['level']) : null;
            $price   = is_null($acc['price']) ? null : $acc['price'];
            $captionText = $acc['caption'] ?: '';

            $lines = [];
            $lines[] = "<blockquote><b>🧾 {$title}</b></blockquote>";
            if ($level) {
                $lines[] = "<blockquote><b>📶 Level: <code>{$level}</code></b></blockquote>";
            }
            if ($captionText) {
                $lines[] = "<blockquote expandable><b>" . htmlspecialchars($captionText) . "</b></blockquote>";
            }
            if ($price !== null) {
                $lines[] = "<blockquote><b>💰 Price: <code>" . format_inr_label($price) . "</code></b></blockquote>";
            }
            $lines[] = "\n <b>Tap BUY to reserve this account. Payments are final — pay & verify.</b>";
            $caption = implode("\n", array_filter($lines));

            $buy_label = $price !== null ? "💳 BUY [" . format_inr_label($price) . "]" : "💳 BUY";
            $kb = ['inline_keyboard' => [[['text' => $buy_label, 'callback_data' => "buy_{$acc['id']}"]]]];

            $resp = apiRequest('sendPhoto', [
                'chat_id' => $chat_id,
                'photo' => $img_url,
                'caption' => $caption,
                'parse_mode' => 'HTML',
                'reply_markup' => json_encode($kb)
            ]);

            if (is_array($resp)) {
                $respData = $resp;
            } elseif (is_string($resp)) {
                $respData = json_decode($resp, true);
            } else {
                $respData = [];
            }

            if (isset($respData['ok']) && $respData['ok'] && isset($respData['result']['message_id'])) {
                $sent_msg_ids[] = (int)$respData['result']['message_id'];
            }
        }

        $next_page = $page + 1;
        $more_row = [['text' => '🔽 More','callback_data' => "cat_{$cat_id}_page_{$next_page}"]];
        if ($owner_link) $more_row[] = ['text' => '🧾 Directly Purchase', 'url' => $owner_link];
        $kb2 = ['inline_keyboard' => [$more_row]];

        $moreResp = apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "🔽 Tap More for more in this category.",
            'reply_markup' => json_encode($kb2)
        ]);

        if (is_array($moreResp)) {
            $moreData = $moreResp;
        } elseif (is_string($moreResp)) {
            $moreData = json_decode($moreResp, true);
        } else {
            $moreData = [];
        }

        if (isset($moreData['ok']) && $moreData['ok'] && isset($moreData['result']['message_id'])) {
            $sent_msg_ids[] = (int)$moreData['result']['message_id'];
        }

        if ($user && isset($user['id'])) {
            saveBrowseSession($pdo, $user['id'], $chat_id, $cat_id, $page, $sent_msg_ids);
        }

        apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id']]);
        exit;
    }

    // buy flow: buy_{account_id}
    if (preg_match('#^buy_(\d+)$#', $data, $m)) {
        $acc_id = (int)$m[1];

        if ($user && isset($user['id'])) {
            cleanupBrowseSession($pdo, $user['id'], $chat_id, $msg_id); // keep only this card
        }

        try {
            $stmt = $pdo->prepare("SELECT * FROM accounts WHERE id = ?");
            $stmt->execute([$acc_id]);
            $acc = $stmt->fetch();
        } catch (PDOException $e) {
            error_log("DB ERROR fetch account: " . $e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Internal error.']);
            exit;
        }

        if (!$acc) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '❌ Account not found.']);
            exit;
        }
        if ($acc['is_sold']) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '❌ Already sold.']);
            exit;
        }

        try {
            $stmt = $pdo->prepare("INSERT INTO orders (user_id, account_id, status) VALUES (?,?, 'pending')");
            $stmt->execute([$user['id'], $acc_id]);
            $order_id = $pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("DB ERROR create order: " . $e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Could not create order.']);
            exit;
        }

        $buyer_chat = $cb['from']['id'];
        $amount = isset($acc['price']) && $acc['price'] !== null ? number_format((float)$acc['price'], 2, '.', '') : "1.00";
        $txn_ref = "ORD" . $order_id;
        $upi_id = isset($config['paytm_upi_id']) ? $config['paytm_upi_id'] : (isset($config['PAYTM_UPI_ID']) ? $config['PAYTM_UPI_ID'] : null);

        $upiString = "upi://pay?pa=" . rawurlencode($upi_id) . "&pn=" . rawurlencode($config['site_name'] ?? 'Merchant') . "&am={$amount}&tr={$txn_ref}&tn=" . rawurlencode("Purchase Order #{$order_id}");

        $kb = ['inline_keyboard'=>[
            [['text'=>'✅ Verify Payment','callback_data'=>"verifypay_{$order_id}"]],
            [['text'=>'❌ Cancel','callback_data'=>"cancel_{$order_id}"]]
        ]];

        // send QR
        if ($has_qrcode_lib) {
            $tmpdir = __DIR__ . '/temp';
            if (!is_dir($tmpdir)) @mkdir($tmpdir, 0755, true);
            $qrPath = $tmpdir . '/' . $txn_ref . '.png';
            QRcode::png($upiString, $qrPath, QR_ECLEVEL_L, 8);

            if (!file_exists($qrPath)) {
                error_log("QR generation failed for $qrPath");
                apiRequest('answerCallbackQuery', [
                    'callback_query_id' => $cb['id'],
                    'text' => '⚠️ Could not create QR. Try again later.',
                    'show_alert' => true
                ]);
            } else {
                $token = isset($config['bot_token']) ? $config['bot_token'] : null;
                if (!$token) {
                    error_log("Bot token missing in config");
                    apiRequest('answerCallbackQuery', [
                        'callback_query_id' => $cb['id'],
                        'text' => '⚠️ Bot token not configured.',
                        'show_alert' => true
                    ]);
                } else {
                    $post = [
                        'chat_id' => $buyer_chat,
                        'caption' => "<b>Pay ₹{$amount} via UPI QR\nThen tap Verify Payment</b>",
                        'parse_mode' => 'HTML',
                        'reply_markup' => json_encode($kb),
                        'photo' => new CURLFile($qrPath)
                    ];
                    $ch = curl_init("https://api.telegram.org/bot{$token}/sendPhoto");
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $post);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

                    $resp = curl_exec($ch);
                    $curlErr = curl_error($ch);
                    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                    curl_close($ch);

                    if ($curlErr) {
                        error_log("sendPhoto curl err: $curlErr");
                        apiRequest('answerCallbackQuery', [
                            'callback_query_id' => $cb['id'],
                            'text' => '⚠️ Could not send QR. Try again.',
                            'show_alert' => true
                        ]);
                    } elseif ($httpCode < 200 || $httpCode >= 300) {
                        error_log("sendPhoto unexpected HTTP code: $httpCode resp: " . substr($resp, 0, 1000));
                        apiRequest('answerCallbackQuery', [
                            'callback_query_id' => $cb['id'],
                            'text' => '⚠️ Telegram refused the image. Check bot token & file size.',
                            'show_alert' => true
                        ]);
                    } else {
                        @unlink($qrPath);

                        $respArr = json_decode($resp, true);
                        if (isset($respArr['ok']) && $respArr['ok'] && isset($respArr['result']['message_id'])) {
                            $qrMid = (int)$respArr['result']['message_id'];

                            try {
                                $up = $pdo->prepare("UPDATE orders SET qr_message_id = ? WHERE id = ?");
                                $up->execute([$qrMid, $order_id]);
                            } catch (PDOException $e) {
                                error_log("DB ERROR save qr_message_id: " . $e->getMessage());
                            }

                        }

                        apiRequest('answerCallbackQuery', [
                            'callback_query_id' => $cb['id'],
                            'text' => '🧾 QR sent — tap Verify Payment after paying.'
                        ]);
                    }
                }
            }
        } else {
            $text = "<b>Open your UPI app and pay:</b>\n\nAmount: ₹{$amount}\nUPI ID: <code>{$upi_id}</code>\nNote/TxnRef: <code>{$txn_ref}</code>\n\nOr scan an external QR generated from the string below.\n\n<code>{$upiString}</code>\n\nAfter payment, tap Verify Payment.";
            $resp = apiRequest('sendMessage', [
                'chat_id' => $buyer_chat,
                'text' => $text,
                'parse_mode' => 'HTML',
                'reply_markup' => json_encode($kb)
            ]);

            if (is_array($resp)) {
                $dataResp = $resp;
            } elseif (is_string($resp)) {
                $dataResp = json_decode($resp, true);
            } else {
                $dataResp = [];
            }

            if (isset($dataResp['ok']) && $dataResp['ok'] && isset($dataResp['result']['message_id'])) {
                $qrMid = (int)$dataResp['result']['message_id'];

                try {
                    $up = $pdo->prepare("UPDATE orders SET qr_message_id = ? WHERE id = ?");
                    $up->execute([$qrMid, $order_id]);
                } catch (PDOException $e) {
                    error_log("DB ERROR save qr_message_id (no QR lib): " . $e->getMessage());
                }

            }
        }

        exit;
    }

    // verify payment flow using Paytm merchant-status for UPI transactions
    if (preg_match('#^verifypay_(\d+)$#', $data, $m)) {
        $order_id = (int)$m[1];

        try {
            $stmt = $pdo->prepare("SELECT 
                                       o.*, 
                                       u.id AS user_db_id,
                                       u.telegram_id AS buyer_tg, 
                                       u.username AS buyer_username, 
                                       u.first_name, 
                                       u.last_name, 
                                       a.title AS account_title, 
                                       a.image_path, 
                                       a.username_value, 
                                       a.password_value, 
                                       a.id AS accid, 
                                       a.is_sold, 
                                       a.level, 
                                       a.gmail
                                   FROM orders o
                                   JOIN users u ON o.user_id = u.id
                                   JOIN accounts a ON o.account_id = a.id
                                   WHERE o.id = ?");
            $stmt->execute([$order_id]);
            $row = $stmt->fetch();
        } catch (PDOException $e) {
            error_log("DB ERROR verifypay fetch: " . $e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Internal error.']);
            exit;
        }

        if (!$row) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '❌ Order not found.', 'show_alert' => true]);
            exit;
        }

        $caller_tg = $from['id'];
        if ((int)$caller_tg !== (int)$row['buyer_tg']) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⛔ You are not the buyer for this order.', 'show_alert' => true]);
            exit;
        }

        if ($row['status'] === 'paid' || (int)$row['is_sold'] === 1) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => 'ℹ️ This order or account was already processed.', 'show_alert' => true]);
            exit;
        }

        // paytm config
        $PAYTM_MID = isset($config['PAYTM_MERCHANT_ID']) ? $config['PAYTM_MERCHANT_ID'] : (isset($config['paytm_merchant_id']) ? $config['paytm_merchant_id'] : null);
        $PAYTM_KEY = isset($config['PAYTM_MERCHANT_KEY']) ? $config['PAYTM_MERCHANT_KEY'] : (isset($config['paytm_merchant_key']) ? $config['paytm_merchant_key'] : null);

        if (!$PAYTM_MID || !$PAYTM_KEY) {
            error_log("Paytm credentials missing in config");
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Payment gateway not configured.', 'show_alert' => true]);
            exit;
        }

        $txn_ref = "ORD" . $order_id;
        $payload = [
            "MID"     => $PAYTM_MID,
            "ORDERID" => $txn_ref
        ];
        $json = json_encode($payload);
        $checksum = hash_hmac("sha256", $json, $PAYTM_KEY);

        $paytm_env = isset($config['paytm_env']) ? $config['paytm_env'] : 'production';
        if ($paytm_env === 'staging' || $paytm_env === 'test') {
            $url = "https://securegw-stage.paytm.in/merchant-status/getTxnStatus?JsonData=" . urlencode($json) . "&CHECKSUMHASH=" . $checksum;
        } else {
            $url = "https://securegw.paytm.in/merchant-status/getTxnStatus?JsonData=" . urlencode($json) . "&CHECKSUMHASH=" . $checksum;
        }

        $resp_raw = @file_get_contents($url);
        if ($resp_raw === false) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Could not verify payment at this time. Try again.', 'show_alert' => true]);
            exit;
        }
        $resp = json_decode($resp_raw, true);

        if (isset($resp["STATUS"]) && $resp["STATUS"] === "TXN_SUCCESS") {
            // mark paid + sold
            try {
                $pdo->prepare("UPDATE orders SET status='paid', updated_at = NOW() WHERE id = ?")->execute([$order_id]);
                $pdo->prepare("UPDATE accounts SET is_sold=1 WHERE id = ?")->execute([$row['accid']]);
            } catch (PDOException $e) {
                error_log("DB ERROR verifypay update: " . $e->getMessage());
                apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Internal error.']);
                exit;
            }

            // Clean QR + old error message
            if (!empty($row['qr_message_id'])) {
                apiRequest('deleteMessage', [
                    'chat_id'    => $row['buyer_tg'],
                    'message_id' => (int)$row['qr_message_id']
                ]);
            }
            if (!empty($row['last_error_message_id'])) {
                apiRequest('deleteMessage', [
                    'chat_id'    => $row['buyer_tg'],
                    'message_id' => (int)$row['last_error_message_id']
                ]);
            }

            // drop the selected card (browse session)
            cleanupBrowseSession($pdo, (int)$row['user_id'], (int)$row['buyer_tg'], null);

            // send creds
            global $config;
            $img_url = $config['site_url'] . '/uploads/' . basename($row['image_path']);
            
            $acc_title = htmlspecialchars($row['account_title'] ?: 'Account');
            $cred_user = htmlspecialchars($row['username_value'] ?? '—');
            $cred_pass = htmlspecialchars($row['password_value'] ?? '—');
            $level     = $row['level'] ? htmlspecialchars($row['level']) : null;
            $gmail     = $row['gmail'] ? htmlspecialchars($row['gmail']) : null;
            
            $captionLines = [];
            $captionLines[] = "<blockquote><b>🎉 {$acc_title}</b></blockquote>";
            if ($level) {
                $captionLines[] = "<blockquote><b>𝗟𝗘𝗩𝗘𝗟: <code>{$level}</code></b></blockquote>";
            }
            $captionLines[] = "<blockquote><b>𝕏 𝗧𝗪𝗜𝗧𝗧𝗘𝗥</b></blockquote>";
            $captionLines[] = "<blockquote><b>𝗨𝗦𝗘𝗥𝗡𝗔𝗠𝗘: <code>{$cred_user}</code></b></blockquote>";
            $captionLines[] = "<blockquote><b>𝗣𝗔𝗦𝗦𝗪𝗢𝗥𝗗: <code>{$cred_pass}</code></b></blockquote>";
            if ($gmail) {
                $captionLines[] = "<blockquote><b>𝗚𝗠𝗔𝗜𝗟: <code>{$gmail}</code></b></blockquote>";
            }
            $captionLines[] = "\n<b>✅ Payment verified. Enjoy your account — keep these credentials safe.</b>";
            
            $cred_caption = implode("\n", $captionLines);
            
            $keyboard = [
                [
                    ['text' => "🔒 Secure Gmail", 'url' => 'https://t.me/fakemailbot'],
                    ['text' => "➡️ Go to Login",   'url' => 'https://x.com/i/flow/login']
                ],
                [
                    ['text' => "📩 Contact Owner", 'url' => 'https://t.me/Ceo_DarkFury']
                ]
            ];
            $reply_markup = json_encode(['inline_keyboard' => $keyboard]);
            
            $respSend = apiRequest('sendPhoto', [
                'chat_id' => $row['buyer_tg'],
                'photo'   => $img_url,
                'caption' => $cred_caption,
                'parse_mode' => 'HTML',
                'reply_markup' => $reply_markup
            ]);

            if (is_array($respSend)) {
                $sendData = $respSend;
            } elseif (is_string($respSend)) {
                $sendData = json_decode($respSend, true);
            } else {
                $sendData = [];
            }

            if (isset($sendData['ok']) && $sendData['ok'] && isset($sendData['result']['message_id'])) {
                $credMid = (int)$sendData['result']['message_id'];

            }

            // owner notify (same as before)
            $buyer_name = trim(($row['first_name'] ?? '') . ' ' . ($row['last_name'] ?? ''));
            $buyer_name = $buyer_name ?: ($row['buyer_username'] ?? '—');
            
            $owner_text_lines = [];
            $owner_text_lines[] = "<blockquote><b>🔔 Sale Completed</b></blockquote>";
            $owner_text_lines[] = "<blockquote><b>🧾 Order:</b> <code>#{$order_id}</code></blockquote>";
            $owner_text_lines[] = "<blockquote><b>👤 Buyer:</b> {$buyer_name}</blockquote>";
            $owner_text_lines[] = "<blockquote><b>🆔 Buyer TG:</b> <code>{$row['buyer_tg']}</code></blockquote>";
            $owner_text_lines[] = "<blockquote><b>📦 Account:</b> <b>{$acc_title}</b></blockquote>";
            if ($level) {
                $owner_text_lines[] = "<blockquote><b>📶 Level:</b> <code>{$level}</code></blockquote>";
            }
            $owner_text_lines[] = "<blockquote><b>🪪 Account ID:</b> <code>{$row['accid']}</code></blockquote>";
            $owner_text_lines[] = "<b>✅ Credentials sent to buyer.</b>";
            $owner_text = implode("\n", $owner_text_lines);

            sendPhoto($config['owner_chat_id'], $img_url, $owner_text, null);

            apiRequest('answerCallbackQuery', [
                'callback_query_id' => $cb['id'],
                'text' => "✅ Payment confirmed.\nCredentials sent successfully.",
                'show_alert' => true
            ]);
            exit;
        } else {
            // FAILED / NOT RECEIVED
            // clean previous "payment not received" message if any
            if (!empty($row['last_error_message_id'])) {
                apiRequest('deleteMessage', [
                    'chat_id'    => $row['buyer_tg'],
                    'message_id' => (int)$row['last_error_message_id']
                ]);
            }

            apiRequest('answerCallbackQuery', [
                'callback_query_id' => $cb['id'],
                'text' => "❌ Payment not received.\nTry again after a moment.",
                'show_alert' => true
            ]);
        
            $text = "❌ Payment not received. Please try Verify again after a moment.";
            $errResp = apiRequest('sendMessage', [
                'chat_id' => $caller_tg,
                'text'    => $text
            ]);

            if (is_array($errResp)) {
                $errData = $errResp;
            } elseif (is_string($errResp)) {
                $errData = json_decode($errResp, true);
            } else {
                $errData = [];
            }

            if (isset($errData['ok']) && $errData['ok'] && isset($errData['result']['message_id'])) {
                $errMid = (int)$errData['result']['message_id'];

                try {
                    $up = $pdo->prepare("UPDATE orders SET last_error_message_id = ? WHERE id = ?");
                    $up->execute([$errMid, $order_id]);
                } catch (PDOException $e) {
                    error_log("DB ERROR save last_error_message_id: " . $e->getMessage());
                }
            }

            exit;
        }
    }

    // CANCEL PAYMENT: cancel_{order_id}
    if (preg_match('#^cancel_(\d+)$#', $data, $m)) {
        $order_id = (int)$m[1];

        try {
            $stmt = $pdo->prepare("SELECT 
                                       o.*, 
                                       u.telegram_id AS buyer_tg
                                   FROM orders o
                                   JOIN users u ON o.user_id = u.id
                                   WHERE o.id = ?");
            $stmt->execute([$order_id]);
            $row = $stmt->fetch();
        } catch (PDOException $e) {
            error_log("DB ERROR cancel fetch: " . $e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Internal error.']);
            exit;
        }

        if (!$row) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '❌ Order not found.']);
            exit;
        }

        $caller_tg = $from['id'];
        if ((int)$caller_tg !== (int)$row['buyer_tg']) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⛔ You are not the buyer for this order.', 'show_alert' => true]);
            exit;
        }

        if ($row['status'] !== 'pending') {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => 'ℹ️ This order is not active anymore.']);
            exit;
        }

        if (!empty($row['qr_message_id'])) {
            apiRequest('deleteMessage', [
                'chat_id'    => $row['buyer_tg'],
                'message_id' => (int)$row['qr_message_id']
            ]);
        }
        if (!empty($row['last_error_message_id'])) {
            apiRequest('deleteMessage', [
                'chat_id'    => $row['buyer_tg'],
                'message_id' => (int)$row['last_error_message_id']
            ]);
        }

        try {
            $pdo->prepare("UPDATE orders SET status='cancelled', qr_message_id=NULL, last_error_message_id=NULL, updated_at = NOW() WHERE id = ?")->execute([$order_id]);
        } catch (PDOException $e) {
            error_log("DB ERROR cancel update: " . $e->getMessage());
        }

        apiRequest('answerCallbackQuery', [
            'callback_query_id'=> $cb['id'],
            'text' => '❌ Payment cancelled.'
        ]);
        exit;
    }

    // === AUTO-VERIFY old route (kept for compatibility) ===
    if (preg_match('#^verify_(\d+)$#', $data, $m)) {
        $order_id = (int)$m[1];
        apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => 'ℹ️ Use the Verify Payment button on the QR message to confirm payment.']);
        exit;
    }

    // remove_{order_id} - hides purchased item
    if (preg_match('#^remove_(\d+)$#', $data, $m)) {
        $order_id = (int)$m[1];
        try {
            $stmt = $pdo->prepare("SELECT user_id FROM orders WHERE id = ?");
            $stmt->execute([$order_id]);
            $o = $stmt->fetch();
        } catch (PDOException $e) {
            error_log("DB ERROR remove fetch: ".$e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Internal error.']);
            exit;
        }
        if (!$o) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '❌ Order not found.']);
            exit;
        }
        $userRow = $user;
        if (!$userRow) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ User error.']);
            exit;
        }
        if ((int)$o['user_id'] !== (int)$userRow['id']) {
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⛔ You can only remove your own purchases.']);
            exit;
        }
        try {
            $stmt = $pdo->prepare("UPDATE orders SET visible = 0 WHERE id = ? AND user_id = ?");
            $stmt->execute([$order_id, $userRow['id']]);
        } catch (PDOException $e) {
            error_log("DB ERROR remove update: ".$e->getMessage());
            apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '⚠️ Could not remove.']);
            exit;
        }
        apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id'], 'text' => '🗑️ Removed from your purchased list.']);
        @apiRequest('editMessageCaption', [
            'chat_id' => $chat_id,
            'message_id' => $msg_id,
            'caption' => "✅ Removed from your purchased list."
        ]);
        exit;
    }

    apiRequest('answerCallbackQuery', ['callback_query_id'=> $cb['id']]);
    exit;
}